#include <windows.h>
#include "resource.h"
#include <Richedit.h>
#include <commctrl.h>
#include <gdiplus.h>

#pragma comment(lib, "comctl32.lib")
#pragma comment(lib, "gdiplus.lib")

using namespace Gdiplus;

// Global Variables:
ULONG_PTR gdiplusToken;
HINSTANCE hInst;                                // current instance
WCHAR szTitle[] = L"かのあゆジャーナル";                  // The title bar text
WCHAR szWindowClass[] = L"DIARYAPP";            // the main window class name

// Forward declarations of functions included in this code module:
ATOM                MyRegisterClass(HINSTANCE hInstance);
BOOL                InitInstance(HINSTANCE, int);
LRESULT CALLBACK    WndProc(HWND, UINT, WPARAM, LPARAM);
void                InsertImageFromFile(HWND hEdit, HWND hWnd);
void                ChangeFontSize(HWND hEdit, int delta);
int CALLBACK        EnumFontFamExProc(const LOGFONTW* lpelfe, const TEXTMETRICW* lpntme, DWORD FontType, LPARAM lParam);
void                ApplySelectedFont(HWND hEdit, HWND hFontCombo);

int APIENTRY wWinMain(_In_ HINSTANCE hInstance,
                     _In_opt_ HINSTANCE hPrevInstance,
                     _In_ LPWSTR    lpCmdLine,
                     _In_ int       nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    // Initialize GDI+
    GdiplusStartupInput gdiplusStartupInput;
    GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);

    // Initialize common controls for the toolbar.
    INITCOMMONCONTROLSEX icex;
    icex.dwSize = sizeof(INITCOMMONCONTROLSEX);
    icex.dwICC = ICC_BAR_CLASSES; // For TOOLBARCLASSNAME
    InitCommonControlsEx(&icex);

    // Load the rich edit library.
    LoadLibrary(TEXT("Msftedit.dll"));

    // Initialize global strings
    MyRegisterClass(hInstance);

    // Perform application initialization:
    if (!InitInstance (hInstance, nCmdShow))
    {
        return FALSE;
    }

    MSG msg;

    // Main message loop:
    while (GetMessage(&msg, nullptr, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }

    GdiplusShutdown(gdiplusToken);

    return (int) msg.wParam;
}

ATOM MyRegisterClass(HINSTANCE hInstance)
{
    WNDCLASSEXW wcex;

    wcex.cbSize = sizeof(WNDCLASSEX);

    wcex.style          = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc    = WndProc;
    wcex.cbClsExtra     = 0;
    wcex.cbWndExtra     = 0;
    wcex.hInstance      = hInstance;
    wcex.hIcon          = LoadIcon(nullptr, IDI_APPLICATION);
    wcex.hCursor        = LoadCursor(nullptr, IDC_ARROW);
    wcex.hbrBackground  = (HBRUSH)(COLOR_WINDOW+1);
    wcex.lpszMenuName   = MAKEINTRESOURCEW(IDC_DIARYAPP);
    wcex.lpszClassName  = szWindowClass;
    wcex.hIconSm        = LoadIcon(nullptr, IDI_APPLICATION);

    return RegisterClassExW(&wcex);
}

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
   hInst = hInstance; // Store instance handle in our global variable

   HWND hWnd = CreateWindowW(szWindowClass, szTitle, WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, nullptr, nullptr, hInstance, nullptr);

   if (!hWnd)
   {
      return FALSE;
   }

   ShowWindow(hWnd, nCmdShow);
   UpdateWindow(hWnd);

   return TRUE;
}

// Callback function for EM_STREAMOUT
static DWORD CALLBACK EditStreamOutCallback(DWORD_PTR dwCookie, LPBYTE pbBuff, LONG cb, PLONG pcb)
{
    HANDLE hFile = (HANDLE)dwCookie;
    DWORD dwWritten;
    if (!WriteFile(hFile, pbBuff, cb, &dwWritten, NULL))
        return -1; // Error
    *pcb = dwWritten;
    return 0; // Success
}

// Function to save the content of the rich edit control to a file
BOOL SaveFile(HWND hEdit, HWND hWnd)
{
    OPENFILENAMEW ofn;
    WCHAR szFileName[MAX_PATH] = L"";

    // Generate default filename from current date
    SYSTEMTIME st;
    GetLocalTime(&st);
    wsprintfW(szFileName, L"%d-%02d-%02d.rtf", st.wYear, st.wMonth, st.wDay);

    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = hWnd;
    ofn.lpstrFilter = L"リッチテキスト形式 (*.rtf)\0*.rtf\0すべてのファイル (*.*)\0*.*\0";
    ofn.lpstrFile = szFileName;
    ofn.nMaxFile = MAX_PATH;
    ofn.Flags = OFN_EXPLORER | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY | OFN_OVERWRITEPROMPT;
    ofn.lpstrDefExt = L"rtf";

    if (GetSaveFileNameW(&ofn))
    {
        HANDLE hFile = CreateFileW(szFileName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
        if (hFile != INVALID_HANDLE_VALUE)
        {
            EDITSTREAM es;
            es.dwCookie = (DWORD_PTR)hFile;
            es.pfnCallback = EditStreamOutCallback;
            SendMessage(hEdit, EM_STREAMOUT, SF_RTF, (LPARAM)&es);
            CloseHandle(hFile);
            return TRUE; // Success
        }
    }
    return FALSE; // Failure or Cancelled
}

// Callback function for EM_STREAMIN
static DWORD CALLBACK EditStreamInCallback(DWORD_PTR dwCookie, LPBYTE pbBuff, LONG cb, PLONG pcb)
{
    HANDLE hFile = (HANDLE)dwCookie;
    DWORD dwRead;
    if (!ReadFile(hFile, pbBuff, cb, &dwRead, NULL))
        return -1; // Error
    *pcb = dwRead;
    if (*pcb == 0) return 1; // End of file
    return 0; // Success
}

BOOL OpenFile(HWND hEdit, HWND hWnd)
{
    OPENFILENAMEW ofn;
    WCHAR szFileName[MAX_PATH] = L"";

    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = hWnd;
    ofn.lpstrFilter = L"リッチテキスト形式 (*.rtf)\0*.rtf\0すべてのファイル (*.*)\0*.*\0";
    ofn.lpstrFile = szFileName;
    ofn.nMaxFile = MAX_PATH;
    ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
    ofn.lpstrDefExt = L"rtf";

    if (GetOpenFileNameW(&ofn))
    {
        HANDLE hFile = CreateFileW(szFileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);
        if (hFile != INVALID_HANDLE_VALUE)
        {
            EDITSTREAM es;
            es.dwCookie = (DWORD_PTR)hFile;
            es.pfnCallback = EditStreamInCallback;
            SendMessage(hEdit, EM_STREAMIN, SF_RTF, (LPARAM)&es);
            CloseHandle(hFile);
            return TRUE; // Success
        }
    }
    return FALSE; // Failure or Cancelled
}

// Helper function to toggle a character effect (bold, italic, underline)
void ToggleCharEffect(HWND hEdit, DWORD dwMask, DWORD dwEffect)
{
    CHARFORMAT2W cf;
    cf.cbSize = sizeof(cf);
    cf.dwMask = dwMask;
    // Get current selection's format
    SendMessage(hEdit, EM_GETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);
    // Toggle the effect
    cf.dwEffects ^= dwEffect;
    // Apply the new format
    SendMessage(hEdit, EM_SETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);
}

// Function to show the font dialog and apply the selected font
void ChooseFontAndApply(HWND hEdit, HWND hWnd)
{
    CHOOSEFONTW cf;
    LOGFONTW lf;
    CHARFORMAT2W cf2;

    // Get current character format to initialize the dialog
    cf2.cbSize = sizeof(cf2);
    SendMessage(hEdit, EM_GETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf2);

    // Initialize LOGFONT from CHARFORMAT2
    GetObjectW(GetStockObject(DEFAULT_GUI_FONT), sizeof(LOGFONTW), &lf);
    if ((cf2.dwMask & CFM_FACE))
    {
        lstrcpyW(lf.lfFaceName, cf2.szFaceName);
    }
    if ((cf2.dwMask & CFM_SIZE))
    {
        // Convert twips to logical units. A simple approximation.
        lf.lfHeight = -MulDiv(cf2.yHeight, 72, 1440);
    }
    lf.lfWeight = (cf2.dwEffects & CFE_BOLD) ? FW_BOLD : FW_NORMAL;
    lf.lfItalic = (cf2.dwEffects & CFE_ITALIC) ? TRUE : FALSE;
    lf.lfUnderline = (cf2.dwEffects & CFE_UNDERLINE) ? TRUE : FALSE;

    // Initialize CHOOSEFONT
    ZeroMemory(&cf, sizeof(cf));
    cf.lStructSize = sizeof(cf);
    cf.hwndOwner = hWnd;
    cf.lpLogFont = &lf;
    cf.Flags = CF_INITTOLOGFONTSTRUCT | CF_SCREENFONTS | CF_EFFECTS;
    cf.rgbColors = cf2.crTextColor;

    if (ChooseFontW(&cf))
    {
        // User clicked OK, create a CHARFORMAT2 from the LOGFONT
        ZeroMemory(&cf2, sizeof(cf2));
        cf2.cbSize = sizeof(cf2);
        cf2.dwMask = CFM_FACE | CFM_SIZE | CFM_BOLD | CFM_ITALIC | CFM_UNDERLINE | CFM_COLOR;

        lstrcpyW(cf2.szFaceName, lf.lfFaceName);
        // Convert logical units back to twips.
        cf2.yHeight = -MulDiv(lf.lfHeight, 1440, 72);
        cf2.dwEffects = 0;
        if (lf.lfWeight >= FW_BOLD) cf2.dwEffects |= CFE_BOLD;
        if (lf.lfItalic) cf2.dwEffects |= CFE_ITALIC;
        if (lf.lfUnderline) cf2.dwEffects |= CFE_UNDERLINE;
        cf2.crTextColor = cf.rgbColors;

        // Apply the new format to the selection
        SendMessage(hEdit, EM_SETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf2);
    }
}

// Callback function for font enumeration
int CALLBACK EnumFontFamExProc(const LOGFONTW* lpelfe, const TEXTMETRICW* lpntme, DWORD FontType, LPARAM lParam)
{
    HWND hFontCombo = (HWND)lParam;
    // Add font name to combobox, avoiding duplicates
    if (SendMessage(hFontCombo, CB_FINDSTRINGEXACT, -1, (LPARAM)lpelfe->lfFaceName) == CB_ERR)
    {
        SendMessage(hFontCombo, CB_ADDSTRING, 0, (LPARAM)lpelfe->lfFaceName);
    }
    return 1; // Continue enumeration
}

// Applies the currently selected font from the combobox to the rich edit selection
void ApplySelectedFont(HWND hEdit, HWND hFontCombo)
{
    int selectedIndex = SendMessage(hFontCombo, CB_GETCURSEL, 0, 0);
    if (selectedIndex != CB_ERR)
    {
        WCHAR fontName[LF_FACESIZE];
        SendMessage(hFontCombo, CB_GETLBTEXT, selectedIndex, (LPARAM)fontName);

        CHARFORMAT2W cf;
        cf.cbSize = sizeof(cf);
        cf.dwMask = CFM_FACE;
        lstrcpyW(cf.szFaceName, fontName);
        SendMessage(hEdit, EM_SETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);
    }
}

// Helper function to change the font size of the selection
void ChangeFontSize(HWND hEdit, int delta)
{
    CHARFORMAT2W cf;
    cf.cbSize = sizeof(cf);
    // Get current selection's format. We need all the info to preserve it.
    SendMessage(hEdit, EM_GETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);

    // We only want to change the size, so set the mask accordingly.
    cf.dwMask = CFM_SIZE;

    // yHeight is in twips (1/1440 of an inch). 20 twips = 1 point.
    // Add the delta. Ensure it doesn't go below a minimum size (e.g., 20 twips for 1pt).
    if (cf.dwMask & CFM_SIZE)
    {
        cf.yHeight += delta;
        if (cf.yHeight < 20) 
        {
            cf.yHeight = 20;
        }
    }
    else // If no font size is set for the selection, start from a default
    {
        cf.yHeight = 240; // 12pt
        cf.yHeight += delta;
    }

    // Apply the new format
    SendMessage(hEdit, EM_SETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);
}

// Function to insert an image from a file into the rich edit control
void InsertImageFromFile(HWND hEdit, HWND hWnd)
{
    OPENFILENAMEW ofn;
    WCHAR szFileName[MAX_PATH] = L"";

    ZeroMemory(&ofn, sizeof(ofn));
    ofn.lStructSize = sizeof(ofn);
    ofn.hwndOwner = hWnd;
    ofn.lpstrFilter = L"Image Files (*.bmp;*.jpg;*.jpeg;*.png;*.gif)\0*.bmp;*.jpg;*.jpeg;*.png;*.gif\0All Files (*.*)\0*.*\0";
    ofn.lpstrFile = szFileName;
    ofn.nMaxFile = MAX_PATH;
    ofn.Flags = OFN_EXPLORER | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;

    if (GetOpenFileNameW(&ofn))
    {
        Bitmap bitmap(szFileName);
        if (bitmap.GetLastStatus() == Ok)
        {   
            HBITMAP hBitmap = NULL;
            // The second parameter is the background color for transparency. NULL means default.
            bitmap.GetHBITMAP(Color(0,0,0,0), &hBitmap);

            if (OpenClipboard(hWnd))
            {
                EmptyClipboard();
                SetClipboardData(CF_BITMAP, hBitmap);
                CloseClipboard();

                SendMessage(hEdit, WM_PASTE, 0, 0);
            }
            // Note: hBitmap is now owned by the clipboard, so we don't delete it here.
            // GDI+ Bitmap object and hBitmap will be cleaned up automatically.
        }
        else
        {
            MessageBoxW(hWnd, L"Could not load the selected image.", L"Image Error", MB_OK | MB_ICONERROR);
        }
    }
} 

// Message handler for about box.
INT_PTR CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    UNREFERENCED_PARAMETER(lParam);
    switch (message)
    {
    case WM_INITDIALOG:
        return (INT_PTR)TRUE;

    case WM_COMMAND:
        if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL)
        {
            EndDialog(hDlg, LOWORD(wParam));
            return (INT_PTR)TRUE;
        }
        break;
    }
    return (INT_PTR)FALSE;
}

LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static HWND hEdit;
    static HWND hToolbar;
    static HWND hFontCombo;
    static BOOL isModified = FALSE;

    switch (message)
    {
    case WM_CREATE:
        {
            // Create Toolbar
            hToolbar = CreateWindowEx(0, TOOLBARCLASSNAME, NULL,
                WS_CHILD | WS_VISIBLE | TBSTYLE_TOOLTIPS, 0, 0, 0, 0,
                hWnd, (HMENU)IDC_TOOLBAR, (HINSTANCE)GetWindowLongPtr(hWnd, GWLP_HINSTANCE), NULL);

            if (hToolbar == NULL)
            {
                MessageBox(hWnd, L"Could not create toolbar.", L"Error", MB_OK | MB_ICONERROR);
                break;
            }

            SendMessage(hToolbar, TB_BUTTONSTRUCTSIZE, (WPARAM)sizeof(TBBUTTON), 0);

            // Create and set the image list for the toolbar
            HIMAGELIST hImageList = ImageList_Create(16, 16, ILC_COLOR24 | ILC_MASK, 5, 0);
            HBITMAP hBitmap = LoadBitmap((HINSTANCE)GetWindowLongPtr(hWnd, GWLP_HINSTANCE), MAKEINTRESOURCE(IDB_TOOLBAR));
            
            // The background color of the bitmap (grey) will be treated as transparent.
            ImageList_AddMasked(hImageList, hBitmap, RGB(192, 192, 192));
            DeleteObject(hBitmap);

            SendMessage(hToolbar, TB_SETIMAGELIST, 0, (LPARAM)hImageList);

            TBBUTTON tbButtons[] =
            {
                { 0, IDM_FORMAT_BOLD,      TBSTATE_ENABLED, BTNS_BUTTON, {0}, 0, 0 },
                { 1, IDM_FORMAT_ITALIC,    TBSTATE_ENABLED, BTNS_BUTTON, {0}, 0, 0 },
                { 2, IDM_FORMAT_UNDERLINE, TBSTATE_ENABLED, BTNS_BUTTON, {0}, 0, 0 },
                { 3, IDM_FORMAT_FONT_INC,  TBSTATE_ENABLED, BTNS_BUTTON, {0}, 0, 0 },
                { 4, IDM_FORMAT_FONT_DEC,  TBSTATE_ENABLED, BTNS_BUTTON, {0}, 0, 0 }
            };

            SendMessage(hToolbar, TB_ADDBUTTONS, (WPARAM)(sizeof(tbButtons)/sizeof(TBBUTTON)), (LPARAM)&tbButtons);

            // Create Font ComboBox
            hFontCombo = CreateWindowEx(0, L"ComboBox", NULL,
                WS_CHILD | WS_VISIBLE | CBS_DROPDOWNLIST | WS_VSCROLL,
                250, 2, 150, 200, // Temporary position/size
                hWnd, (HMENU)IDC_FONT_COMBO, (HINSTANCE)GetWindowLongPtr(hWnd, GWLP_HINSTANCE), NULL);

            // Populate the font combobox
            LOGFONTW lf;
            lf.lfCharSet = DEFAULT_CHARSET;
            lf.lfFaceName[0] = L'\0';
            lf.lfPitchAndFamily = 0;
            HDC hdc = GetDC(hWnd);
            EnumFontFamiliesExW(hdc, &lf, (FONTENUMPROCW)EnumFontFamExProc, (LPARAM)hFontCombo, 0);
            ReleaseDC(hWnd, hdc);

            // Create Rich Edit Control
            hEdit = CreateWindowEx(
                0, MSFTEDIT_CLASS,   // Rich Edit Class
                L"",        // No edit box text
                WS_CHILD | WS_VISIBLE | WS_VSCROLL |
                ES_LEFT | ES_MULTILINE | ES_AUTOVSCROLL | ES_NOHIDESEL,
                0, 0, 0, 0, // Set size in WM_SIZE
                hWnd,         // Parent window
                (HMENU)IDC_MAIN_EDIT,
                (HINSTANCE)GetWindowLongPtr(hWnd, GWLP_HINSTANCE),
                nullptr);

            if (hEdit == NULL)
            {
                MessageBox(hWnd, L"Could not create rich edit control.", L"Error", MB_OK | MB_ICONERROR);
                break;
            }
            
            SendMessage(hEdit, EM_SETEVENTMASK, 0, (LPARAM)ENM_CHANGE);
        }
        break;
    case WM_SIZE:
        {
            SendMessage(hToolbar, TB_AUTOSIZE, 0, 0);

            RECT rcToolbar;
            GetWindowRect(hToolbar, &rcToolbar);
            int toolbarHeight = rcToolbar.bottom - rcToolbar.top;

            // Get the right edge of the last toolbar button to position the combobox
            RECT rcLastBtn;
            SendMessage(hToolbar, TB_GETRECT, (WPARAM)2, (LPARAM)&rcLastBtn); // Index 2 is the 3rd button

            int comboX = rcLastBtn.right + 8;
            int comboY = (toolbarHeight - 24) / 2; // Vertically center (assuming ~24px height for the combo)
            
            MoveWindow(hFontCombo, comboX, comboY, 150, 200, TRUE);

            MoveWindow(hEdit,
                0, toolbarHeight,
                LOWORD(lParam),
                HIWORD(lParam) - toolbarHeight,
                TRUE);
        }
        break;
    case WM_COMMAND:
        {
            int wmId = LOWORD(wParam);
            int wmEvent = HIWORD(wParam);

            if (wmId == IDC_MAIN_EDIT && wmEvent == EN_CHANGE)
            {
                isModified = TRUE;
            }

            switch (wmId)
            {
            case IDC_FONT_COMBO:
                if (wmEvent == CBN_SELCHANGE)
                {
                    ApplySelectedFont(hEdit, hFontCombo);
                }
                break;
            case IDM_ABOUT:
                DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUTBOX), hWnd, About);
                break;
            case IDM_EXIT:                DestroyWindow(hWnd);
                break;
            case IDM_FILE_OPEN:
                if (OpenFile(hEdit, hWnd))
                {
                    isModified = FALSE;
                }
                break;
            case IDM_FILE_SAVE_AS:
                if (SaveFile(hEdit, hWnd))
                {
                    isModified = FALSE;
                }
                break;
            case IDM_EDIT_PASTE:
                SendMessage(hEdit, WM_PASTE, 0, 0);
                break;
            case IDM_EDIT_INSERT_IMAGE:
                InsertImageFromFile(hEdit, hWnd);
                break;
            case IDM_FORMAT_BOLD:
                ToggleCharEffect(hEdit, CFM_BOLD, CFE_BOLD);
                break;
            case IDM_FORMAT_ITALIC:
                ToggleCharEffect(hEdit, CFM_ITALIC, CFE_ITALIC);
                break;
            case IDM_FORMAT_UNDERLINE:
                ToggleCharEffect(hEdit, CFM_UNDERLINE, CFE_UNDERLINE);
                break;
            case IDM_FORMAT_FONT_INC:
                ChangeFontSize(hEdit, 20); // Increase by 1pt
                break;
            case IDM_FORMAT_FONT_DEC:
                ChangeFontSize(hEdit, -20); // Decrease by 1pt
                break;
            case IDM_FORMAT_FONT:
                ChooseFontAndApply(hEdit, hWnd);
                break;
            default:
                return DefWindowProc(hWnd, message, wParam, lParam);
            }
        }
        break;
    case WM_CLOSE:
        if (isModified)
        {
            int result = MessageBoxW(hWnd, L"内容が変更されています。保存しますか？", L"日記アプリ", MB_YESNOCANCEL | MB_ICONWARNING);
            if (result == IDYES)
            {
                if (SaveFile(hEdit, hWnd))
                {
                    DestroyWindow(hWnd);
                }
            }
            else if (result == IDNO)
            {
                DestroyWindow(hWnd);
            }
        }
        else
        {
            DestroyWindow(hWnd);
        }
        break;
    case WM_DESTROY:
        PostQuitMessage(0);
        break;
    default:
        return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}
